module net.BurtonRadons.parse.expression;

//debug = Semantic;

/** Base expression class. */
class Expression
{
    import net.BurtonRadons.parse.error;
    import net.BurtonRadons.parse.lexer;
    import net.BurtonRadons.parse.scope;
    import net.BurtonRadons.parse.type;
    
    Marker mark;
    Type type; /**< The resultant type of the expression, or null before the semantic phase for most types. */

    /** Implicit cast these two to be compatible and return the resulting type. */
    static Type implicitCast (Scope scope, inout Expression a, inout Expression b)
    {
        if (a.type != b.type)
        {
            Type type = a.type.implicitCastFor (b.type);

            if (type === null)
                type = b.type.implicitCastFor (a.type);
            assert (type !== null);

            a = a.implicitCastTo (scope, type);
            b = b.implicitCastTo (scope, type);
            return type;
        }

        assert (a.type == b.type);
        return a.type;
    }

    /** Implicit cast this expression to the type or return null. */
    Expression implicitCastTo (Scope scope, Type to)
    {
        if (type == to)
            return this;
        if (cast (BasicType) type && cast (BasicType) to)
            return new CastExp (mark, to, this);
        scope.semanticError (new CannotImplicitCastError (mark, type.toString (), to.toString ()));
        return this;
    }

    /** Register names in the parent scope. */
    Expression semantic0 (Scope scope)
    {
        return this;
    }

    /** Perform symbol linking. */
    Expression semantic1 (Scope scope)
    {
        return this;
    }

    /** Confirm code integrity. */
    Expression semantic2 (Scope scope)
    {
        return this;
    }

    char [] toString ()
    {
        return toString (false);
    }

    char [] toString (bit nested)
    {
        throw new Error (this.classinfo.name ~ ".toString () unimplemented\n");
        return null;
    }
}

/** An integer constant. */
class IntegerExp : Expression
{
    ulong value; /**< Constant value. */

    /** Assign parameters. */
    this (Marker mark, ulong value, Type type)
    {
        this.mark = mark;
        this.value = value;
        this.type = type;
    }

    override char [] toString (bit nested)
    {
        return fmt ("%Lu", value);
    }

    override Expression implicitCastTo (Scope scope, Type to)
    {
        if (cast (FloatType) to || cast (DoubleType) to || cast (RealType) to)
            return new FloatExp (mark, value, to);
        return super.implicitCastTo (scope, to);
    }
}

/** A floating-point constant. */
class FloatExp : Expression
{
    real value; /**< Constant value. */

    /** Assign parameters. */
    this (Marker mark, real value, Type type)
    {
        this.mark = mark;
        this.value = value;
        this.type = type;
    }
}

/** An imaginary constant. */
class ImaginaryExp : Expression
{
    real value; /**< Constant value. */

    /** Assign parameters. */
    this (Marker mark, real value)
    {
        this.mark = mark;
        this.value = value;
    }
}

/** The "true" constant. */
class TrueExp : IntegerExp
{
    /** Assign the parameter and setup the parent. */
    this (Marker mark)
    {
        super (mark, 1, BitType.singleton);
    }
}

/** The "false" constant. */
class FalseExp : IntegerExp
{
    /** Assign the parameter and setup the parent. */
    this (Marker mark)
    {
        super (mark, 0, BitType.singleton);
    }
}

/** A string constant. */
class StringExp : Expression
{
    char [] value; /**< The string composing the constant. */

    /** Assign parameters. */
    this (Marker mark, char [] value)
    {
        this.mark = mark;
        this.value = value;
        this.type = new ArrayType (CharType.singleton);
    }

    Expression implicitCastTo (Scope scope, Type to)
    {
        if ((cast (PointerType) to && cast (CharType) to.next)
         || (cast (ArrayType) to && cast (CharType) to.next))
        {
            this.type = to;
            return this;
        }

        return super.implicitCastTo (scope, to);
    }

}

/** An identifier expression. */
class IdExp : Expression
{
    import net.BurtonRadons.parse.declaration;
    
    char [] name; /**< Name of the identifier. */
    Symbol symbol; /**< The symbol this has been determined to be of. */

    /** Assign parameters. */
    this (Marker mark, char [] name)
    {
        this.mark = mark;
        this.name = name;
    }

    override Expression semantic1 (Scope scope)
    {
        symbol = scope.find (name);
        if (symbol === null)
            scope.semanticError (new UndefinedIdentifierError (mark, name));
        else
        {
            type = symbol.type;

            /* Convert to a "this.id" if this is referring to an aggregate value. */
            Declaration decl = symbol.decl;

            if (cast (VariableDeclaration) decl && (cast (VariableDeclaration) decl).aggr !== null)
            {
                decl = (cast (VariableDeclaration) decl).aggr;
                goto convertToThis;
            }

            if (decl !== null && decl.scope !== null && decl.scope.parent !== null && (cast (FunctionDeclaration) decl || cast (VariableDeclaration) decl))
            {
                decl = decl.scope.parent.findRoot ();

            convertToThis:
                if (cast (AggregateDeclaration) decl)
                {
                    ThisExp thisExp = new ThisExp (mark);

                    thisExp.type = (cast (AggregateDeclaration) decl).name.type;
                    DotIdExp exp = new DotIdExp (thisExp, name);

                    exp.symbol = symbol;
                    return exp.semantic1 (scope);
                }
            }
        }

        return this;
    }

    char [] toString (bit nested) { return name; }
}

/** The "this" reference. */
class ThisExp : Expression
{
    import net.BurtonRadons.parse.declaration;
    
    /** Assign the parameter. */
    this (Marker mark)
    {
        this.mark = mark;
        type = VoidType.singleton;
    }

    char [] toString () { return "this"; }

    Expression semantic0 (Scope scope)
    {
        Declaration root = scope.findRoot ();

        if (root === null || (cast (FunctionDeclaration) root) === null)
        {
            scope.semanticError (new ThisOutsideFunctionError (mark));
            return this;
        }

        FunctionDeclaration func = cast (FunctionDeclaration) root;

        if (func.aggr === null || func.isStatic)
        {
            scope.semanticError (new NoThisInFunctionError (mark, func.name.toString ()));
            return this;
        }

        assert (func !== null);
        type = func.aggr.name.type;
        return this;
    }
}

/** Return the virtual object of the superclass, such as to call a super-method. */
class SuperExp : Expression
{
    /** Assign the parameter. */
    this (Marker mark)
    {
        this.mark = mark;
    }
}

/** Reference to the "null" virtual constant. */
class NullExp : Expression
{
    /** Assign the parameter. */
    this (Marker mark, Type type)
    {
        this.mark = mark;
        this.type = type;
    }
}

/** A function call expression; "a (args)". */
class CallExp : Expression
{
    Expression a; /**< Expression for the function to call, null for a special function. */
    Expression [] args; /**< Arguments applied to the call. */
    FunctionType ftype;

    /** Assign parameters. */
    this (Expression a, Expression [] args)
    {
        this.a = a;
        this.args = args;
        if (a !== null)
            this.mark = a.mark;
    }

    /** Return the number of bytes the arguments take.  Post-semantic. */
    int argsCellSize ()
    {
        int size;

        for (int c; c < args.length; c ++)
        {
            if (args [c].type !== null)
                size += args [c].type.cellSize ();
        }

        return size;
    }

    override Expression semantic0 (Scope scope)
    {
        debug (Semantic) printf ("CallExp.semantic0\n");
        if (a !== null)
            a = a.semantic0 (scope);
        for (int c; c < args.length; c ++)
            args [c] = args [c].semantic0 (scope);
        if (type !== null)
            type = type.semantic0 (scope);
        return this;
    }

    override Expression semantic1 (Scope scope)
    {
        debug (Semantic) printf ("CallExp.semantic1\n");

        if (a !== null)
        {
            a = a.semantic1 (scope);
            type = a.type.next;
            ftype = cast (FunctionType) a.type;
        }

        if (type !== null)
            type = type.semantic1 (scope);
        if (args.length < ftype.args.length)
            throw new Error ("Too few arguments for function (ERROR TODO)");
        if (args.length > ftype.args.length && !ftype.varargs)
            throw new Error ("Too many arguments for function (ERROR TODO)");

        /* Call the semantic phase on the functions and to implicit type casting. */
        for (int c; c < ftype.args.length; c ++)
        {
            Type to = ftype.args [c].type;
            Expression result;

            if (ftype.decl.castFloatToDouble () && cast (FloatType) to)
                to = DoubleType.singleton;

            args [c] = args [c].semantic1 (scope);
            if (args [c].type === null)
                continue;
            args [c] = args [c].implicitCastTo (scope, to);
        }

        for (int c = ftype.args.length; c < args.length; c ++)
        {
            args [c] = args [c].semantic1 (scope);
            if (ftype.decl.castFloatToDouble () && cast (FloatType) args [c].type)
                args [c] = args [c].implicitCastTo (scope, DoubleType.singleton);
        }

        return this;
    }

    override Expression semantic2 (Scope scope)
    {
        debug (Semantic) printf ("CallExp.semantic2\n");
        if (a !== null)
            a = a.semantic2 (scope);
        if (type !== null)
            type = type.semantic2 (scope);
        for (int c; c < args.length; c ++)
            args [c] = args [c].semantic2 (scope);
        return this;
    }

    override char [] toString (bit nested)
    {
        char [] string = a.toString (true) ~ " (";

        for (int c; c < args.length; c ++)
        {
            if (c) string ~= ", ";
            string ~= args [c].toString (false);
        }

        return string ~ ")";
    }
}

/** Allocate an array or a class instance. */
class NewExp : CallExp
{
    /** Assign parameters. */
    this (Marker mark, Type type, Expression [] args)
    {
        super (null, args);
        this.mark = mark;
        this.type = type;
        this.ftype = new FunctionType (VoidType.singleton, null);
    }
}

/** An explicit cast expression; "cast (type) a". */
class CastExp : Expression
{
    Expression a; /**< The expression being casted. */

    /** Assign parameters. */
    this (Marker mark, Type type, Expression a)
    {
        this.mark = mark;
        this.type = type;
        this.a = a;
    }

    Expression semantic0 (Scope scope)
    {
        a = a.semantic0 (scope);
        type = type.semantic0 (scope);
        return this;
    }

    Expression semantic1 (Scope scope)
    {
        a = a.semantic1 (scope);
        type = type.semantic1 (scope);
        return this;
    }

    Expression semantic2 (Scope scope)
    {
        a = a.semantic2 (scope);
        type = type.semantic2 (scope);
        return this;
    }
}

/** A "(at).id" expression. */
class TypeDotIdExp : Expression
{
    Type at; /**< The type being accessed. */
    char [] id; /**< The identifier name. */

    /** Assign parameters. */
    this (Marker mark, Type at, char [] id)
    {
        this.mark = mark;
        this.at = at;
        this.id = id;
    }

    override Expression semantic0 (Scope scope)
    {
        at = at.semantic0 (scope);
        return this;
    }

    override Expression semantic1 (Scope scope)
    {
        at = at.semantic1 (scope);
        return this;
    }

    override Expression semantic2 (Scope scope)
    {
        at = at.semantic2 (scope);

        if (id == "size")
            return new IntegerExp (mark, at.cellSize (), IntType.singleton);
        throw new Error (fmt ("Error TODO: Type %.*s has no property %.*s.", at.toString (), id));
        return this;
    }
}

/** Slice an array; "array [start .. end]". */
class SliceExp : Expression
{
    Expression array; /**< Array to slice. */
    Expression start; /**< Starting index or null for 0. */
    Expression end; /**< Ending index (inclusive) or null for the end of the array. */

    /** Assign parameters. */
    this (Expression array, Expression start, Expression end)
    {
        this.mark = array.mark;
        this.array = array;
        this.start = start;
        this.end = end;
    }
}

/** Access the field of an object; "a.id". */
class DotIdExp : Expression
{
    import net.BurtonRadons.parse.declaration;
    
    Expression a; /**< Object to access the field of. */
    char [] id; /**< Identifier to access. */
    Symbol symbol; /**< Symbol linked to after the semantic phase.  */

    /** Assign parameters. */
    this (Expression a, char [] id)
    {
        this.a = a;
        this.id = id;
        this.mark = a.mark;
    }

    override Expression semantic0 (Scope scope)
    {
        a = a.semantic0 (scope);
        return this;
    }

    override Expression semantic1 (Scope scope)
    {
        a = a.semantic1 (scope);

        if (id == "size")
            return new IntegerExp (mark, a.type.cellSize (), IntType.singleton);

        if (cast (ArrayType) a.type)
        {
            if (id == "length")
                return new ArrayLengthExp (mark, a);
            else
                assert (0);
        }
        else if (cast (StructType) a.type || cast (ClassType) a.type)
        {
            AggregateDeclaration decl;

            if (cast (StructType) a.type)
                decl = (cast (StructType) a.type).decl;
            else if (cast (ClassType) a.type)
                decl = (cast (ClassType) a.type).decl;
            else
                assert (0);

            if (id in decl.scope.decls)
            {
                symbol = decl.scope.decls [id];
                type = symbol.type;
            }
            else
                assert (0);
        }
        else
            throw new Error ("DotIdExp.semantic1 type " ~ a.type.classinfo.name);

        return this;
    }

    override Expression semantic2 (Scope scope)
    {
        a = a.semantic2 (scope);
        return this;
    }
}

/** Return the length of an array. */
class ArrayLengthExp : Expression
{
    Expression a; /**< Array object to access. */

    /** Assign parameters. */
    this (Marker mark, Expression a)
    {
        this.mark = mark;
        this.a = a;
        this.type = UIntType.singleton;
        assert (cast (ArrayType) a.type);
    }

    override Expression semantic2 (Scope scope)
    {
        a = a.semantic2 (scope);
        return this;
    }
}

/** An individual item for an ArrayExp. */
struct ArrayItem
{
    Expression index; /**< Index or null to use the last index plus one.  Zero-based. */
    Expression value; /**< Value of this item. */
}

/** An on-stack array declaration. */
class ArrayExp : Expression
{
    ArrayItem [] list; /**< Items in the array. */

    /** Assign parameters. */
    this (Marker mark, ArrayItem [] list)
    {
        this.mark = mark;
        this.list = list;
    }
}

/** An on-stack struct declaration. */
class StructExp : Expression
{
    /** An individual item in the declaration. */
    struct Item
    {
        char [] name; /**< Name of the field or null. */
        Symbol symbol; /**< The field this has been attached to or null. */
        Expression value; /**< The value of the item. */
    }

    Item [] list; /**< Declaration values. */

    /** Assign the parameter. */
    this (Marker mark)
    {
        this.mark = mark;
    }

    /** Add an item. */
    void add (Symbol symbol, Expression value)
    {
        Item item;

        item.name = symbol.name;
        item.symbol = symbol;
        item.value = value;
        list ~= item;
    }

    /** Find an item's value; gets the init value if there is none. */
    Expression value (Symbol symbol)
    {
        for (int c; c < list.length; c ++)
            if (list [c].symbol === symbol)
                return list [c].value;

        if (symbol.init !== null)
            return symbol.init;
        return symbol.type.field (mark, "init");
    }
}

/** If test evaluates to true, evaluate a, else evaluate b. */
class ConditionalExp : Expression
{
    Expression test; /**< The value to test. */
    Expression onTrue; /**< Executed if test is true. */
    Expression onFalse; /**< Executed if test is false. */

    /** Assign parameters. */
    this (Marker mark, Expression test, Expression onTrue, Expression onFalse)
    {
        this.mark = mark;
        this.test = test;
        this.onTrue = onTrue;
        this.onFalse = onFalse;
    }
}

/** Under debug builds, test for this expression to be true; if false, throw an assertion failure. */
class AssertExp : Expression
{
    Expression test; /**< The value to test. */

    /** Assign parameters. */
    this (Marker mark, Expression test)
    {
        this.type = VoidType.singleton;
        this.mark = mark;
        this.test = test;
    }

    Expression semantic0 (Scope scope) { test = test.semantic0 (scope); return this; }
    Expression semantic1 (Scope scope) { test = test.semantic1 (scope); return this; }

    Expression semantic2 (Scope scope)
    {
        test = test.semantic2 (scope);
        test = test.implicitCastTo (scope, BitType.singleton);
        return this;
    }
}

/** A unary expression. */
class UnaryExp : Expression
{
    Lexer.TokenType op; /**< Operative token type. */
    Expression a; /**< Expression this applies to. */

    /** Assign parameters. */
    this (Marker mark, Lexer.TokenType op, Expression a)
    {
        this.mark = mark;
        this.op = op;
        this.a = a;
    }

    override Expression semantic0 (Scope scope) { a = a.semantic0 (scope); type = a.type; return this; }
    override Expression semantic1 (Scope scope) { a = a.semantic1 (scope); type = a.type; return this; }
    override Expression semantic2 (Scope scope) { a = a.semantic2 (scope); type = a.type; return this; }
}

/** A unary expression for prefixed operators ("++ a", for example). */
class PreUnaryExp : UnaryExp
{
    this (Marker mark, Lexer.TokenType op, Expression a)
    {
        super (mark, op, a);
    }
}

/** Unary negate (-a). */
class NegExp : PreUnaryExp
{
    /** Assign the parameters. */
    this (Marker mark, Expression a)
    {
        super (mark, Lexer.subToken, a);
    }
}

/** Post-increment operator (a ++). */
class PostIncExp : UnaryExp
{
    /** Assign the parameter. */
    this (Expression a)
    {
        super (a.mark, Lexer.addAddToken, a);
    }
}

/** Post-decrement operator (a --). */
class PostDecExp : UnaryExp
{
    /** Assign the parameter. */
    this (Expression a)
    {
        super (a.mark, Lexer.subSubToken, a);
    }
}

/** Pre-increment operator (++ a). */
class PreIncExp : PreUnaryExp
{
    /** Assign the parameters. */
    this (Marker mark, Expression a)
    {
        super (mark, Lexer.addAddToken, a);
    }
}

/** Pre-decrement operator (-- a). */
class PreDecExp : PreUnaryExp
{
    /** Assign the parameters. */
    this (Marker mark, Expression a)
    {
        super (mark, Lexer.subSubToken, a);
    }
}

/** Retrieve the pointer to a value; "&a". */
class AddressExp : PreUnaryExp
{
    /** Assign parameters. */
    this (Marker mark, Expression a)
    {
        super (mark, Lexer.andToken, a);
    }
}

/** Access the value at the pointer; "*a". */
class PointerExp : PreUnaryExp
{
    /** Assign parameters. */
    this (Marker mark, Expression a)
    {
        super (mark, Lexer.mulToken, a);
    }
}

/** Convert the parameter to boolean and invert its value. */
class LogicalNotExp : PreUnaryExp
{
    /** Assign parameters. */
    this (Marker mark, Expression a)
    {
        super (mark, Lexer.notToken, a);
    }
}

/** A binary expression. */
class BinaryExp : UnaryExp
{
    Expression b; /**< Right side of the expression. */

    /** Assign parameters. */
    this (Expression a, Lexer.TokenType op, Expression b)
    {
        this.a = a;
        this.op = op;
        this.b = b;
        this.mark = a.mark.contain (b.mark);
        super (this.mark, op, a);
    }

    override Expression semantic0 (Scope scope)
    {
        a = a.semantic0 (scope);
        b = b.semantic0 (scope);
        return this;
    }

    override Expression semantic1 (Scope scope)
    {
        a = a.semantic1 (scope);
        b = b.semantic1 (scope);
        if (a.type === null || b.type === null)
            return this;
        type = implicitCast (scope, a, b);
        return this;
    }

    override Expression semantic2 (Scope scope)
    {
        a = a.semantic2 (scope);
        b = b.semantic2 (scope);
        return this;
    }

    char [] toString (bit nested)
    { 
        if (nested)
            return "(" ~ a.toString () ~ " " ~ Lexer.tokenTypeRepr (op) ~ " " ~ b.toString () ~ ")";
        else
            return a.toString () ~ " " ~ Lexer.tokenTypeRepr (op) ~ " " ~ b.toString ();
    }
}

/** Execute a and b and return b. */
class CommaExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.commaToken, b);
    }
}

/** One of the assignment expressions (specified through the op). */
class AssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Lexer.TokenType op, Expression b)
    {
        super (a, op, b);
    }

    override Expression semantic1 (Scope scope)
    {
        a = a.semantic1 (scope);
        b = b.semantic1 (scope);
        if (a.type === null || b.type === null)
            return this;
        type = a.type;
        b = b.implicitCastTo (scope, type);
        return this;
    }
}

/** Addition (a + b). */
class AddExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.addToken, b);
    }
}

/** Addition and assignment (a += b). */
class AddAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.addAssToken, b);
    }
}

/** Subtraction (a - b). */
class SubExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.subToken, b);
    }
}

/** Subtraction and assignment (a -= b). */
class SubAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.subAssToken, b);
    }
}

/** Multiplication (a * b). */
class MulExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.mulToken, b);
    }
}

/** Multiplication and assignment (a *= b). */
class MulAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.mulAssToken, b);
    }
}

/** Division (a / b). */
class DivExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.divToken, b);
    }
}

/** Division and assignment (a /= b). */
class DivAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.divAssToken, b);
    }
}

/** Return the remainder of a / b (a % b). */
class ModExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.divToken, b);
    }
}

/** Assign the remainder of a / b (a %= b). */
class ModAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.divAssToken, b);
    }
}

/** Index an array (a [b]).  Index is zero-based. */
class IndexExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.lbracketToken, b);
    }
}

/** Equals (a == b) or not opEquals (a != b) depending upon op. */
class EqualsExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Lexer.TokenType op, Expression b)
    {
        super (a, op, b);
        type = BitType.singleton;
    }

    override Expression semantic0 (Scope scope)
    {
        a = a.semantic0 (scope);
        b = b.semantic0 (scope);
        return this;
    }

    override Expression semantic1 (Scope scope)
    {
        a = a.semantic1 (scope);
        b = b.semantic1 (scope);
        implicitCast (scope, a, b);
        return this;
    }

    override Expression semantic2 (Scope scope)
    {
        a = a.semantic2 (scope);
        b = b.semantic2 (scope);
        return this;
    }
}

/** Identity comparison; (a === b) or (a !== b) depending upon op. */
class IdentityEqualsExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Lexer.TokenType op, Expression b)
    {
        super (a, op, b);
    }
}

/** Comparison operator, depends upon op.  Can be lessToken "<",
  * lequalsToken "<=", greaterToken ">", gequalsToken ">=", ueToken "!<>",
  * unordToken "!<>=", ulToken "!<", uleToken "!<=", ugToken "!>", or
  * ugeToken "!>=".
  */

class CompareExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Lexer.TokenType op, Expression b)
    {
        super (a, op, b);
    }

    override Expression semantic1 (Scope scope)
    {
        a = a.semantic1 (scope);
        b = b.semantic1 (scope);
        implicitCast (scope, a, b);
        type = BitType.singleton;
        return this;
    }
}

/** Logical OR operator (a || b). */
class LogicalOrExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.orOrToken, b);
    }
}

/** Logical AND operator (a || b). */
class LogicalAndExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.andAndToken, b);
    }
}

/** Return true if "a" is in "b"; for associative arrays, common arrays. */
class InExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.inToken, b);
    }
}

/** Concatenate "a" and "b" to return "ab". */
class CatExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.catToken, b);
    }
}

/** Concatenate "a" and "b" and assign to "a". */
class CatAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.catAssToken, b);
    }
}

/** Binary left-shift "a << b". */
class LeftShiftExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.lshiftToken, b);
    }
}

/** Binary left shift and assign "a <<= b". */
class LeftShiftAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.lshiftAssToken, b);
    }
}

/** Signed binary right shift "a >> b". */
class RightShiftExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.rshiftToken, b);
    }
}

/** Signed binary right shift and assign "a >>= b". */
class RightShiftAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.rshiftAssToken, b);
    }
}

/** Unsigned binary right-shift "a >>> b". */
class UnsignedRightShiftExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.urshiftToken, b);
    }
}

/** Unsigned binary right shift and assign "a >>>= b". */
class UnsignedRightShiftAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.urshiftAssToken, b);
    }
}

/** Bitwise OR; "a | b". */
class BitwiseOrExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.orToken, b);
    }
}

/** Bitwise OR and assign; "a |= b". */
class BitwiseOrAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.orAssToken, b);
    }
}

/** Bitwise XOR; "a ^ b". */
class BitwiseXorExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.xorToken, b);
    }
}

/** Bitwise XOR and assign; "a ^= b". */
class BitwiseXorAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.xorAssToken, b);
    }
}

/** Bitwise AND; "a & b". */
class BitwiseAndExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.andToken, b);
    }
}

/** Bitwise AND and assign; "a &= b". */
class BitwiseAndAssignExp : BinaryExp
{
    /** Assign parameters. */
    this (Expression a, Expression b)
    {
        super (a, Lexer.andAssToken, b);
    }
}

